<?php

/**
 * This is the model class for table "notification".
 *
 * The followings are the available columns in table 'notification':
 * @property string $id
 * @property string $setting_id id of the row in the notification_setting table
 * @property string $object_id id of the row in notification_setting.object_table table
 * @property string $status status of the object (loading,unread,shown,dismissed)
 * @property string $read date and time of read
 * @property string $epoch date and time of insert
 * @property NotificationSetting $setting
 */
class Notification extends ActiveRecord
{
  const STATUS_LOADING = 'loading';
  const STATUS_UNREAD = 'unread';
  const STATUS_SHOWN = 'shown';
  const STATUS_DISMISSED = 'dismissed';
  const STATUS_DUMPED = 'dumped';

  /***************************/
  /****SEARCH FIELDS**********/
  /***************************/
  public $onlyForCurrentUser = false;
  public $specificSettingId = null;
  /***************************/

  /**
   * Returns the static model of the specified AR class.
   * @param string $className active record class name.
   * @return Notification the static model class
   */
  public static function model($className=__CLASS__)
  {
    return parent::model($className);
  }

  /**
   * @return string the associated database table name
   */
  public function tableName()
  {
    return 'notification';
  }

  /**
   * @return array validation rules for model attributes.
   */
  public function rules()
  {
    // NOTE: you should only define rules for those attributes that
    // will receive user inputs.
    return array(
        array('setting_id, object_id, status, read, epoch', 'required'),
        array('setting_id, object_id', 'length', 'max'=>10),
        array('status', 'length', 'max'=>255),
        // The following rule is used by search().
        // Please remove those attributes that should not be searched.
        array('id, setting_id, object_id, status, read, epoch', 'safe', 'on'=>'search'),
    );
  }

  /**
   * @return array relational rules.
   */
  public function relations()
  {
    // NOTE: you may need to adjust the relation name and the related
    // class name for the relations automatically generated below.
    return array(
        'setting'=>array(self::BELONGS_TO, 'NotificationSetting', 'setting_id'),
    );
  }

  /**
   * @return array customized attribute labels (name=>label)
   */
  public function myAttributeLabels()
  {
    return array(
        'id' => 'ID',
        'setting_id' => t('Setting'),
        'object_id' => t('Object'),
        'status' => t('Status'),
        'read' => t('Read'),
        'epoch' => t('Epoch'),
    );
  }

  /**
   * Retrieves a list of models based on the current search/filter conditions.
   * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
   */
  public function search()
  {
    // Warning: Please modify the following code to remove attributes that
    // should not be searched.
  
    $criteria=new CDbCriteria;
  
    $criteria->compare('id',$this->id,true);
    $criteria->compare('setting_id',$this->setting_id,true);
    $criteria->compare('object_id',$this->object_id,true);
    $criteria->compare('status',$this->status,true);
    $criteria->compare('read',$this->read,true);
    $criteria->compare('epoch',$this->epoch,true);
    
    return new CActiveDataProvider($this, array(
        'criteria'=>$criteria,
    ));
  }

  /**
   * Retrieves a list of models based on the current search/filter conditions.
   * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
   */
  public function searchForGrid()
  {
    // Warning: Please modify the following code to remove attributes that
    // should not be searched.
  
    $criteria=new CDbCriteria;

    if(!isEmpty($this->setting_id))
    {
      $nsids = array();
      foreach (NotificationSetting::getVisibleToUserNotificationSettings() as $setting)
      {
        if(containsStr($setting->description,$this->setting_id))
          $nsids[] = $setting->id;
      }
      $criteria->addInCondition("setting_id", $nsids);
    }

    if(!isEmpty($this->status))
    {
      $nsids = array();
      foreach (Notification::getAllStatus() as $statusKey => $statusLabel)
      {
        if(containsStr($statusLabel,$this->status))
          $nsids[] = $statusKey;
      }
      $criteria->addInCondition("status", $nsids);
    }
    
    if(!isEmpty($this->specificSettingId))
    {
      $criteria->compare("setting_id", $this->specificSettingId);
    }
    
    if($this->onlyForCurrentUser)
    {
      $criteria->with = array("setting");
      $criteria->addCondition("setting.target_id=".user()->id);
    }
  
    $provider = new CActiveDataProvider($this, array(
        'criteria'=>$criteria,
    ));
  
    $provider->pagination->pageSize = 50;
  
    return $provider;
  }
  
  public static function getAllStatus()
  {
    return array(
        self::STATUS_DISMISSED => SystemInfoHelper::getNotificationStatusByString(self::STATUS_DISMISSED),
        self::STATUS_DUMPED => SystemInfoHelper::getNotificationStatusByString(self::STATUS_DUMPED),
        self::STATUS_LOADING => SystemInfoHelper::getNotificationStatusByString(self::STATUS_LOADING),
        self::STATUS_SHOWN => SystemInfoHelper::getNotificationStatusByString(self::STATUS_SHOWN),
        self::STATUS_UNREAD => SystemInfoHelper::getNotificationStatusByString(self::STATUS_UNREAD),
    );
  }
}