<?php

/**
 * This is the model class for table "violation_document".
 *
 * The followings are the available columns in table 'gatetable':
 * @property integer $id                Chiave primaria record
 * @property integer $id_event_behavior         chiave estera verso EventBehavior
 * @property string  $status              stato del documento
 *                              "missing"     documento mancante. è una richiesta di calcolo
 *                              "preview"     documento realizzato come anteprima per la visualizzazione pre-validazione
 *                              "not_available"   documento non disponibile - il sistema non ne prevede la produzione per quel tipo di comportamento    
 *                              "definitive"    documento calcolato in via definitiva nel formato per export finale
 *                              "failed"      calcolo documento fallito
 * @property string  $document              documento salvato in binario su BLOB database (solo se non attiva la modalità che esporta blob come file esterni al database)
 * @property integer $document_size           dimensioni del documento
 * @property integer $document_filename         nome del file dove il documento è esportato
 * @property string  $last_update           istante ultimo aggiornamento
 * @property string  $epoch               istante inserimento
 * 
 * @property EventBehavior  eventBehaviors
 * 
 */
class ViolationDocument extends ActiveRecord
{

  const STATUS_MISSING = 'missing';

  const STATUS_PREVIEW = 'preview';

  const STATUS_NOT_AVAILABLE = 'not_available';

  const STATUS_DEFINITIVE = 'definitive';

  const STATUS_FAILED = 'failed';

  public static function model($className = __CLASS__)
  {
    return parent::model ( $className );
  }

  public function tableName()
  {
    return 'violation_document';
  }

  public function behaviors()
  {
    return array (
        'edatetimebehavior' => array (
            'class' => 'ext.EDateTimeBehavior' 
        ) 
    );
  }

  public function relations()
  {
    return array (
        'eventBehaviors' => array (
            self::BELONGS_TO,
            'EventBehavior',
            'id_event_behavior' 
        ) 
    );
  }

  public function scopes()
  {
    return array (
        'previewAvailable' => array (
            'condition' => 'status = :status',
            'params' => array (
                ':status' => self::STATUS_PREVIEW 
            ) 
        ),
        'definitiveAvailable' => array (
            'condition' => 'status = :status',
            'params' => array (
                ':status' => self::STATUS_DEFINITIVE 
            ) 
        ) 
    );
  }

  /**
   *
   * @return array validation rules for model attributes.
   */
  public function rules()
  {
    // NOTE: you should only define rules for those attributes that
    // will receive user inputs.
    return array (
        array (
            'id_event_behavior, document_size, document_ext_size',
            'length',
            'max' => 20 
        ),
        array (
            'status, document_filename, document_ext_filename',
            'length',
            'max' => 250 
        ),
        array (
            'document, last_update, epoch, document_ext',
            'safe' 
        ),
        // The following rule is used by search().
        // Please remove those attributes that should not be searched.
        array (
            'id, id_event_behavior, status, document, document_size, document_filename, last_update, epoch, document_ext, document_ext_size, document_ext_filename',
            'safe',
            'on' => 'search' 
        ) 
    );
  }

  /**
   *
   * @return array customized attribute labels (name=>label)
   */
  public function myAttributeLabels()
  {
    return array (
        'id' => 'ID',
        'id_event_behavior' => t ( 'Id Event Behavior' ),
        'status' => t ( 'Status' ),
        'document' => t ( 'Document' ),
        'document_size' => t ( 'Document Size' ),
        'document_filename' => t ( 'Document Filename' ),
        'last_update' => t ( 'Last Update' ),
        'epoch' => t ( 'Epoch' ),
        'document_ext' => t ( 'Document Ext' ),
        'document_ext_size' => t ( 'Document Ext Size' ),
        'document_ext_filename' => t ( 'Document Ext Filename' ) 
    );
  }

  public function getUrl()
  {
    $url = Yii::app ()->getRequest ()
      ->getHostInfo ();
    return $url . Yii::app ()->params ['photoBaseUrl'] . "/" . $this->document_filename;
  }

  public function getExtUrl()
  {
    if (empty ( $this->document_ext_filename ))
      return null;
    
    $url = Yii::app ()->getRequest ()
      ->getHostInfo ();
    return $url . Yii::app ()->params ['photoBaseUrl'] . "/" . $this->document_ext_filename;
  }

  /**
   * Retrieves a list of models based on the current search/filter conditions.
   *
   * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
   */
  public function search()
  {
    // Warning: Please modify the following code to remove attributes that
    // should not be searched.
    $criteria = new CDbCriteria ();
    
    $criteria->compare ( 'id', $this->id, true );
    $criteria->compare ( 'id_event_behavior', $this->id_event_behavior, true );
    $criteria->compare ( 'status', $this->status, true );
    $criteria->compare ( 'document', $this->document, true );
    $criteria->compare ( 'document_size', $this->document_size, true );
    $criteria->compare ( 'document_filename', $this->document_filename, true );
    $criteria->compare ( 'last_update', $this->last_update, true );
    $criteria->compare ( 'epoch', $this->epoch, true );
    $criteria->compare ( 'document_ext', $this->document_ext, true );
    $criteria->compare ( 'document_ext_size', $this->document_ext_size, true );
    $criteria->compare ( 'document_ext_filename', $this->document_ext_filename, true );
    
    return new CActiveDataProvider ( $this, array (
        'criteria' => $criteria 
    ) );
  }

  /**
   *
   * @return string
   */
  public function getBaseUrl()
  {
    $url = Yii::app ()->getRequest ()
      ->getHostInfo ();
    return $url . Yii::app ()->params ['photoBaseUrl'] . "/" . $this->document_filename;
  }

  /**
   *
   * @return string
   */
  public function getPath()
  {
    $basePath = Yii::app ()->params ['photoBasePath'];
    $filename = $this->document_filename;
    return pathCombine ( $basePath, $filename );
  }

  /**
   * trova la path locale o url per leggere il file da PHP!
   * 
   * @return string
   */
  public function getFinalSearchPath()
  {
    if (paramdeep ( 'photoBaseLocationByUrl' ))
      $filename = YII_DEBUG ? resolve_redirect ( $this->getBaseUrl () ) : $this->getBaseUrl ();
    else
      $filename = $this->getPath ();
    return $filename;
  }
}
