<?php

/**
 * This is the model class for table "report_archive".
 *
 * The followings are the available columns in table 'report_archive':
 * @property integer $ID
 * @property string $UserID
 * @property string $PeriodID
 * @property string $GroupID
 * @property string $CreationTime
 * @property string $MinExtcode
 * @property string $MaxExtcode
 * @property string $Document
 */
class ReportDocument extends CActiveRecord
{
	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return ReportArchive the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'report_archive';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		// NOTE: you should only define rules for those attributes that
		// will receive user inputs.
		return array(
			array('UserID, PeriodID, GroupID, CreationTime, MinExtcode, MaxExtcode, Document', 'required'),
			array('UserID, PeriodID, GroupID', 'length', 'max'=>20),
			array('MinExtcode, MaxExtcode', 'length', 'max'=>250),
			// The following rule is used by search().
			// Please remove those attributes that should not be searched.
			array('ID, UserID, GroupID, CreationTime, MinExtcode, MaxExtcode, Document', 'safe', 'on'=>'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		// NOTE: you may need to adjust the relation name and the related
		// class name for the relations automatically generated below.
		return array(
			'user' => array(self::BELONGS_TO, "User", "UserID"),
			'period' => array(self::BELONGS_TO, "WorkPeriod", "PeriodID"),
			'gateGroup' => array(self::BELONGS_TO, "GateGroup", "GroupID"),
		);
	}
	
	/**
	 * @return array relational scopes.
	 */
	public function scopes()
	{
		return array(
			'last' => array('order', 'CreationTime DESC'),
		);
	}
	
	public function bySession(WorkPeriod $session)
	{
		$this->getDbCriteria ()
		->mergeWith ( array (
			'condition' => 'PeriodID = :period_id',
			'params' => array (
				':period_id' => $session->id
			)
		) );
		return $this;
	}
	
	public function after($dateTime)
	{
		$this->getDbCriteria()
		->mergeWith(array(
			'condition' => 'CreationTime >= :startDateTime',
			'params' => array(
				':startDateTime' => DateHelper::formatISO9075DateTime($dateTime)
			)
		) );
		return $this;
	}
	
	public function before($dateTime)
	{
		$this->getDbCriteria()
		->mergeWith(array(
			'condition' => 'CreationTime <= :endDateTime',
			'params' => array(
				':endDateTime' => DateHelper::formatISO9075DateTime($dateTime)
			)
		) );
		return $this;
	}
	
	public function byGateGroups($gateGroups)
	{
		$this->getDbCriteria()->addInCondition('GroupID', $gateGroups);
		
		return $this;
	}
	
	public function byUsers($users)
	{
		$this->getDbCriteria()->addInCondition('UserID', $users);
		
		return $this;
	}
	
	public function fromADI($ADIcode)
	{
		$this->getDbCriteria()
		->mergeWith(array(
			'condition' => 'MaxExtcode >= :fromADIcode',
			'params' => array(
				':fromADIcode' => $ADIcode
			)
		) );
		return $this;
	}
	
	public function toADI($ADIcode)
	{
		$this->getDbCriteria()
		->mergeWith(array(
			'condition' => 'MinExtcode <= :toADIcode',
			'params' => array(
				':toADIcode' => $ADIcode
			)
		) );
		return $this;
	}
	
	public function howMany()
	{
		return ReportDocument::model()->count($this->getDbCriteria());
	}
	
	public function behaviors()
	{
		return array (
			'edatetimebehavior' => array (
				'class' => 'ext.EDateTimeBehavior'
			)
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'ID' => 'ID',
			'UserID' => 'User',
			'GroupID' => 'Group',
			'CreationTime' => 'Creation Time',
			'MinExtcode' => 'Min Extcode',
			'MaxExtcode' => 'Max Extcode',
			'Document' => 'Document',
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models
	 *         based on the search/filter conditions.
	 */
	public function search($pageSize)
	{
		$options = array(
			'criteria' => $this->dbCriteria,
		);
		
		if($pageSize) {
			$options['pagination'] = array('pageSize' => $pageSize);
		}
		else {
			$options['pagination'] = false;
		}
		
		return new CActiveDataProvider($this, $options);
	}
	
	public function orderByCreationTime()
	{
		$this->getDbCriteria()->order = 'CreationTime DESC';
		
		return $this;
	}
	
	public function getTitle()
	{
		
		$title = t("Report");
		$title.= "_";
		$title.= $this->user->fullName;
		$title.= "_";
		$title.= new DateTimeEx($this->CreationTime);
		$title.= "_";
		$title.= $this->gateGroup->name;
		$title.= "_";
		$title.= $this->MinExtcode;
		$title.= "_";
		$title.= $this->MaxExtcode;
		
		return $title;
	}
	
	static public function numOfGeneratedReports(WorkPeriod $session)
	{
		$numOfReports = ReportDocument::model()->bySession($session)
			->howMany();
		
		return $numOfReports;
	}
	
}

?>