<?php

/**
 * This is the model class for table "gate_group".
 *
 * The followings are the available columns in table 'gate_group':
 * @property string $ID
 * @property string $name
 * @property string $epoch
 * @property string $latitude
 * @property string $longitude
 * @property string $altitude
 * @property int orderIndex
 * 
 * @property Gate[] $gates
 */
class GateGroup extends ActiveRecord
{
  const STATUS_ALIVE = "ALIVE";
  const STATUS_WARNING = "WARNING";
  const STATUS_DEAD = "DEAD";

  public static function model($className=__CLASS__)
  {
    return parent::model($className);
  }

  public function tableName()
  {
    return 'gate_group';
  }

  public function getConsumers()
  {
    $consumers = array();
    $adder = function(array &$array, $label, $count)
    {
      return array_add_if($array, $count, $count>0, $label, false);
    };
    
    $adder($consumers,'RunSchedule',RunSchedule::model()->count('id_gate_group=:gid', array(':gid'=>$this->ID)));
    $adder($consumers,'Service',Service::model()->count('id_gategroup=:gid', array(':gid'=>$this->ID)));
    $adder($consumers,'Gate',Gate::model()->count('id_group=:gid', array(':gid'=>$this->ID)));

    return $consumers;
  }

  public function rules()
  {
    // NOTE: you should only define rules for those attributes that
    // will receive user inputs.
    return array(
        array('latitude, longitude, altitude', 'length', 'max'=>255),
        array('name', 'length', 'max'=>255),
        array('epoch', 'safe'),
        // The following rule is used by search().
        // Please remove those attributes that should not be searched.
        array('id, name, latitude, longitude, altitude, epoch', 'safe', 'on'=>'search'),
    );
  }

  public function myAttributeLabels()
  {
    return array(
        'name' => t('Name'),
        'altitude' => t('Altitude'),
        'latitude' => t('Latitude'),
        'longitude' => t('Longitude'),
    );
  }


  public function relations()
  {
    return array(
        'gates'              => array(self::HAS_MANY, 'Gate',    'id_group', 'order'=>'Descrizione ASC'),
        'services'           => array(self::HAS_MANY, 'Service', 'id_gategroup'),
        'incompleteServices' => array(self::HAS_MANY, 'Service', 'id_gategroup', 'scopes' => 'incomplete'),
    );
  }

  public function defaultScope()
  {
    $alias = $this->getTableAlias(false, false);

    return array(
        'order' => "$alias.orderIndex",
    );
  }

  public function getAddress()
  {
    if (count($gates = ($this->gates()))>0)
      return $gates[0]->Descrizione;
    else
      return "";
  }

  public function getSpeedLimit()
  {
    if (count($gates = ($this->gates()))>0)
      return $gates[0]->speed_limit;
    else
      return "";
  }

  public function getStatus()
  {
    $gates = $this->gates;
    $statusCount = array_count_values(
        array_map(function($gate) {
          return $gate->Status;
        },
        $gates)
    );

    if (!isset($statusCount[Gate::GS_ALIVE]))
      return self::STATUS_DEAD;

    if ($statusCount[Gate::GS_ALIVE] == count($gates))
      return self::STATUS_ALIVE;
    else
      return self::STATUS_WARNING;
  }

  /**
   * @return string[]
   */
  static public function getGateGroupList()
  {
    $res = GateGroup::model()->findAll();
    if(is_null($res))
      return null;

    $arr = array();
    if(!is_array($res))
      $res = array($res);

    foreach($res as $gt)
      $arr[$gt->ID] = $gt->name;

    return $arr;
  }

  /**
   * @return GateGroup[]
   */
  static public function getGateGroups($sort=true)
  {
    return GateGroup::model()->findAll();
  }

  public function orderByName()
  {
    $this->getDbCriteria()->mergeWith(
        array(
            'order' => 'name ASC',
        )
    );
    return $this;
  }

}
